<?php

namespace Vlabs\MediaBundle\Manager;

use Imagine\Gd\Imagine;
use Imagine\Image\Box;
use Vich\UploaderBundle\Mapping\PropertyMapping;
use Vich\UploaderBundle\Storage\StorageInterface;
use Vlabs\MediaBundle\MediaInterface;
use Vlabs\MediaBundle\Queuing\QueuingChain;
use Vlabs\MediaBundle\Queuing\QueuingInterface;
use Vlabs\MediaBundle\Tools\ExtensionGuesser;

class MediaManager implements MediaManagerInterface
{
    protected QueuingChain $chain;
    protected StorageInterface $storage;
    protected array $config;

    public function __construct(QueuingChain $chain, StorageInterface $storage, array $config = [])
    {
        $this->config  = $config;
        $this->chain   = $chain;
        $this->storage = $storage;
    }

    /**
     * {@inheritdoc}
     */
    public function remapMedia(MediaInterface $media, PropertyMapping $mapping): void
    {
        $file = $media->getMediaFile();

        $media->setPath($mapping->getUriPrefix());
        $media->setMimeType($file->getMimeType());
    }

    /**
     * {@inheritdoc}
     */
    public function publishResize(MediaInterface $media): void
    {
        $queuing = $this->getQueuing($media);

        if (method_exists($queuing, 'setMediaManager')) {
            $queuing->setMediaManager($this);
        }

        foreach ($this->config['resize'][$media->getKey()]['thumbs'] as $thumb => $config) {
            $queuing->enqueue($media, $thumb);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function doResize(MediaInterface $media, $thumb): void
    {
        $imageConfig = $this->config['resize'][$media->getKey()]['thumbs'][$thumb];
        $fsKey       = $this->config['resize'][$media->getKey()]['filesystem'];
        $fs          = $this->storage->getFilesystemFromKey($fsKey);
        $stream      = $fs->readStream(sprintf('%s://%s', $fsKey, $media->getFilename()));

        $imagine = new Imagine();
        $image   = $imagine->load(stream_get_contents($stream));

        $box = new Box($imageConfig['size']['width'], $imageConfig['size']['height']);

        if (isset($imageConfig['relative_resize'])) {
            $box = match ($imageConfig['relative_resize']) {
                'heighten' => $image->getSize()->heighten($imageConfig['size']['height']),
                'widen'    => $image->getSize()->widen($imageConfig['size']['width']),
            };
        }

        $resizedImage = $image->thumbnail($box, $imageConfig['mode'])
                                ->get(ExtensionGuesser::guess($media->getMimeType()));

        $fs->write(sprintf('%s://%s/%s', $fsKey, $imageConfig['uri_prefix'], $media->getFilename()), $resizedImage);
    }

    /**
     * {@inheritdoc}
     */
    public function getQueuing(MediaInterface $media): ?QueuingInterface
    {
        $queuingAlias = $this->config['resize'][$media->getKey()]['queuing'];

        return $this->chain->getQueuing($queuingAlias);
    }

    /**
     * {@inheritdoc}
     */
    public function getUri(MediaInterface $media, bool $absolute = false, ?string $thumb = null): string
    {
        $mediaUri = '';

        if ($absolute) {
            $mediaUri .= isset($this->config['resize'][$media->getKey()]) ? $this->config['resize'][$media->getKey()]['base_url'] : $this->config['default_base_url'];
        }

        $mediaUri .= $media->getPath();

        if ($thumb) {
            $mediaUri .= $this->config['resize'][$media->getKey()]['thumbs'][$thumb]['uri_prefix'];
        }

        return $mediaUri . sprintf('/%s', $media->getFilename());
    }

    /**
     * {@inheritdoc}
     */
    public function hasMedia(MediaInterface $media, ?string $thumb = null): bool
    {
        $fs = $this->storage->getFilesystemFromKey($media->getKey());

        if (null !== $thumb && isset($this->config['resize'][$media->getKey()][$thumb])) {
            $urlPrefix = $this->config['resize'][$media->getKey()]['thumbs'][$thumb]['uri_prefix'];

            return $fs->has(sprintf('%s://%s/%s', $media->getKey(), $urlPrefix, $media->getFilename()));
        }

        return $fs->has(sprintf('%s://%s', $media->getKey(), $media->getFilename()));
    }

    /**
     * {@inheritdoc}
     */
    public function deleteMedia(MediaInterface $media): void
    {
        $fs = $this->storage->getFilesystemFromKey($media->getKey());

        $filePath = sprintf('%s://%s', $media->getKey(), $media->getFilename());

        if ($fs->has($filePath)) {
            $fs->delete($filePath);
        }

        if (isset($this->config['resize'][$media->getKey()])) {
            foreach ($this->config['resize'][$media->getKey()]['thumbs'] as $config) {
                $filePath = sprintf('%s://%s/%s', $media->getKey(), $config['uri_prefix'], $media->getFilename());

                if ($fs->has($filePath)) {
                    $fs->delete($filePath);
                }
            }
        }
    }
}
